#include "stdafx.h"
#include "MessageStatus.h"
#include "PCanIsoTpUtils.h"

// <summary>
/// Constructor.
/// </summary>
/// <param name="p_canTpMsg">The received ISO-TP message.</param>
/// <param name="p_canTpTimestamp">The timestamp when the message was received.</param>
/// <param name="p_listIndex">Position of the messageStatus in the message listView.</param>
MessageStatus::MessageStatus(TPCANTPMsg p_canTpMsg, TPCANTPTimestamp p_canTpTimestamp, int p_listIndex)
{
	m_msg = p_canTpMsg;
	m_timeStamp = p_canTpTimestamp;
	m_timeStampOld = p_canTpTimestamp;
	m_index = p_listIndex;
	m_count = 1;
	m_showPeriod = true;
	m_wasChanged = false;
}

/// <summary>
/// Updates the messageStatus' information with a newly received ISO-TP message.
/// </summary>
/// <param name="p_canTpMsg">The received ISO-TP message.</param>
/// <param name="p_canTpTimestamp">The timestamp when the message was received.</param>
void MessageStatus::update(TPCANTPMsg p_canTpMsg, TPCANTPTimestamp p_canTpTimestamp)
{
	m_msg = p_canTpMsg;
	m_timeStampOld = m_timeStamp;
	m_timeStamp = p_canTpTimestamp;
	m_wasChanged = true;
	m_count++;
}

/// <summary>
/// States if a message has the same network address information as 
/// the last ISO-TP message received.
/// </summary>
/// <param name="p_canTpMsg">The ISO-TP message to compare to.</param>
/// <returns></returns>
bool MessageStatus::isSameNetAddrInfo(TPCANTPMsg p_canTpMsg)
{
	return (m_msg.SA == p_canTpMsg.SA &&
		m_msg.TA == p_canTpMsg.TA &&
		m_msg.RA == p_canTpMsg.RA &&
		m_msg.IDTYPE == p_canTpMsg.IDTYPE &&
		m_msg.FORMAT == p_canTpMsg.FORMAT &&
		m_msg.TA_TYPE == p_canTpMsg.TA_TYPE &&
		m_msg.MSGTYPE == p_canTpMsg.MSGTYPE);
}

/// <summary>
/// The last ISO-TP message received.
/// </summary>
TPCANTPMsg MessageStatus::CanTpMsg() {
	return m_msg;
}

/// <summary>
/// The timestamp of the last ISO-TP message received.
/// </summary>
TPCANTPTimestamp MessageStatus::Timestamp() {
	return m_timeStamp;
}

/// <summary>
/// The index of the object in the "received message" listView.
/// </summary>
int MessageStatus::Position() {
	return m_index;
}

/// <summary>
/// Number of time a similar message was received (same network address information).
/// </summary>
int MessageStatus::Count() {
	return m_count;
}

void MessageStatus::setShowingPeriod(bool p_value) {
	if (m_showPeriod ^ p_value)
	{
		m_showPeriod = p_value;
		m_wasChanged = true;
	}
}

/// <summary>
/// Unique ID as a string of the last message received.
/// </summary>
CString MessageStatus::UID() {
	return CPCanIsoTpUtils::GetUniqueID(m_msg);
}

/// <summary>
/// CAN ID type as a string of the last message received.
/// </summary>
CString MessageStatus::CanIdType() {
	return CPCanIsoTpUtils::GetCanIdType(m_msg.IDTYPE);
}

/// <summary>
/// ISO-TP Message type as a string of the last message received.
/// </summary>
CString MessageStatus::MsgType() {
	return CPCanIsoTpUtils::GetMsgType(m_msg.MSGTYPE);
}

/// <summary>
/// ISO-TP addressing format type as a string of the last message received.
/// </summary>
CString MessageStatus::FormatType() {
	return CPCanIsoTpUtils::GetFormatType(m_msg.FORMAT);
}

/// <summary>
/// ISO-TP target addressing type as a string of the last message received.
/// </summary>
CString MessageStatus::TargetType() {
	return CPCanIsoTpUtils::GetTargetType(m_msg.TA_TYPE);
}

/// <summary>
/// ISO-TP source address as a string of the last message received.
/// </summary>
CString MessageStatus::SourceAddress() {

	if (m_msg.FORMAT == PCANTP_FORMAT_ENHANCED)
		// Unique SA is 11bits: store bits [7..0] in SA
		// and the rest in [3..0] of RA
		return CPCanIsoTpUtils::GetCanId(((UINT32)((m_msg.RA >> 4) & 0x7)) << 8 | m_msg.SA, false);
	return CPCanIsoTpUtils::GetAddress(m_msg.SA);
}

/// <summary>
/// ISO-TP target address as a string of the last message received.
/// </summary>
CString MessageStatus::TargetAddress() {

	if (m_msg.FORMAT == PCANTP_FORMAT_ENHANCED)
		// Unique TA is 11bits: store bits [7..0] in TA
		// and the rest in [7..4] of RA
		return CPCanIsoTpUtils::GetCanId(((UINT32)(m_msg.RA & 0x7)) << 8 | m_msg.TA, false);
	return CPCanIsoTpUtils::GetAddress(m_msg.TA);
}

/// <summary>
/// ISO-TP remote address as a string of the last message received.
/// </summary>
CString MessageStatus::RemoteAddress() {
	return CPCanIsoTpUtils::GetAddress(m_msg.RA);
}

/// <summary>
/// Lenght of the data as a string of the last message received.
/// </summary>
CString MessageStatus::LengthString() {
	CString str;
	str.Format(L"%d", m_msg.LEN);
	return str;
}

/// <summary>
/// Data as a string of the last message received.
/// </summary>
CString MessageStatus::DataString() {
	return CPCanIsoTpUtils::GetDataString(m_msg);
}

/// <summary>
/// ISO-TP Network result as a string of the last message received.
/// </summary>
CString MessageStatus::ResultString() {
	return CPCanIsoTpUtils::GetResult(m_msg.RESULT);
}

/// <summary>
/// Timestamp (or period) as a string of the last message received.
/// </summary>
CString MessageStatus::TimeString() {
	return CPCanIsoTpUtils::GetTimeString(m_timeStamp, m_showPeriod, m_timeStampOld);
}